   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*             CLIPS Version 6.30  10/19/06            */
   /*                                                     */
   /*                   RETRACT MODULE                    */
   /*******************************************************/

/*************************************************************/
/* Purpose:  Handles join network activity associated with   */
/*   with the removal of a data entity such as a fact or     */
/*   instance.                                               */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*      6.24: Removed LOGICAL_DEPENDENCIES compilation flag. */
/*                                                           */
/*            Renamed BOOLEAN macro type to intBool.         */
/*                                                           */
/*            Rule with exists CE has incorrect activation.  */
/*            DR0867                                         */
/*                                                           */
/*      6.30: Added support for hashed alpha memories.       */
/*                                                           */
/*            Added additional developer statistics to help  */
/*            analyze join network performance.              */
/*                                                           */
/*            Removed pseudo-facts used in not CEs.          */
/*                                                           */
/*************************************************************/

#define _RETRACT_SOURCE_

#include <stdio.h>
#define _STDIO_INCLUDED_
#include <stdlib.h>

#include "setup.h"

#if DEFRULE_CONSTRUCT

#include "agenda.h"
#include "argacces.h"
#include "constant.h"
#include "drive.h"
#include "engine.h"
#include "envrnmnt.h"
#include "lgcldpnd.h"
#include "match.h"
#include "memalloc.h"
#include "network.h"
#include "reteutil.h"
#include "router.h"
#include "symbol.h"

#include "retract.h"

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

   static void                    ReturnMarkers(void *,struct multifieldMarker *);
   static void                    DriveRetractions(void *);
   static intBool                 FindNextConflictingAlphaMatch(void *,struct partialMatch *,
                                                                struct partialMatch *,
                                                                struct joinNode *,struct partialMatch *);
   static intBool                 PartialMatchDefunct(void *,struct partialMatch *);
   static void                    PosEntryRetractAlpha(void *,struct partialMatch *,void *);
   static void                    NegEntryRetractAlpha(void *,struct partialMatch *,void *);
   static void                    NegEntryRetractBeta(void *,struct joinNode *,struct partialMatch *,
                                                      struct partialMatch *,void *);
  
/************************************************************/
/* NetworkRetract:  Retracts a data entity (such as a fact  */
/*   or instance) from the pattern and join networks given  */
/*   a pointer to the list of patterns which the data       */
/*   entity matched. The data entity is first removed from  */
/*   the join network through patterns not directly         */
/*   enclosed within a not CE and then through patterns     */
/*   enclosed by a not CE. Any new partial matches created  */
/*   by the removal are then filtered through the join      */
/*   network. This ordering prevents partial matches from   */
/*   being generated that contain the data entity which was */
/*   removed.                                               */
/************************************************************/
globle void NetworkRetract(
  void *theEnv,
  struct patternMatch *listOfMatchedPatterns)
  {
   struct patternMatch *tempMatch, *nextMatch, *lastMatch;

   /*============================================*/
   /* Remove the data entity from all joins that */
   /* aren't directly enclosed by a not CE.      */
   /*============================================*/

   tempMatch = listOfMatchedPatterns;
   lastMatch = NULL;
   while (tempMatch != NULL)
     {
      nextMatch = tempMatch->next;

      PosEntryRetractAlpha(theEnv,tempMatch->theMatch,tempMatch->theMatch->binds[0].gm.theMatch->matchingItem);
      
      /*===================================================*/
      /* Remove from the alpha memory of the pattern node. */
      /*===================================================*/
      
      if (tempMatch->theMatch->children == NULL)
        {
         RemoveAlphaMemoryMatches(theEnv,tempMatch->matchingPattern,
                                  tempMatch->theMatch,
                                  tempMatch->theMatch->binds[0].gm.theMatch,0); /* TBD Alpha Hashing */
      
         if (lastMatch == NULL)
           { listOfMatchedPatterns = nextMatch; }
         else   
           { lastMatch->next = nextMatch; }
           
         rtn_struct(theEnv,patternMatch,tempMatch);
        }
      else
        { lastMatch = tempMatch; }

      tempMatch = nextMatch;
     }

   /*============================================*/
   /* Remove the data entity from all joins that */
   /* are directly enclosed by a not CE.         */
   /*============================================*/
   
   tempMatch = listOfMatchedPatterns;
   while (tempMatch != NULL)
     {
      nextMatch = tempMatch->next;

      NegEntryRetractAlpha(theEnv,tempMatch->theMatch,tempMatch->theMatch->binds[0].gm.theMatch->matchingItem);
      
      RemoveAlphaMemoryMatches(theEnv,tempMatch->matchingPattern,
                                  tempMatch->theMatch,
                                  tempMatch->theMatch->binds[0].gm.theMatch,0); /* TBD Alpha Hashing */
      
      rtn_struct(theEnv,patternMatch,tempMatch);

      tempMatch = nextMatch;
     }

   /*=========================================*/
   /* Filter new partial matches generated by */
   /* retraction through the join network.    */
   /*=========================================*/

   DriveRetractions(theEnv);
  }

/***************************************************************/
/* PosEntryRetractAlpha:           */
/***************************************************************/
static void PosEntryRetractAlpha(
  void *theEnv,
  struct partialMatch *alphaMatch,
  void *duringRetract)
  {
   struct partialMatch *betaMatch, *tempMatch;
   struct joinNode *joinPtr;
   
   betaMatch = alphaMatch->children;
   while (betaMatch != NULL)
     {
      joinPtr = (struct joinNode *) betaMatch->owner;
      if (joinPtr->patternIsNegated)
        {
         betaMatch = betaMatch->nextInRight;
         continue;
        }
              
      if (betaMatch->children != NULL)
        { PosEntryRetractBeta(theEnv,betaMatch,betaMatch->children,duringRetract); }
      
      /* Remove the beta match. */
      
      if ((betaMatch->activationf) ?
          (betaMatch->binds[betaMatch->bcount].gm.theValue != NULL) : FALSE)
           { RemoveActivation(theEnv,(struct activation *) betaMatch->binds[betaMatch->bcount].gm.theValue,TRUE,TRUE); }

      tempMatch = betaMatch->nextInRight;
      
      UnlinkBetaPMFromNodeAndLineage(betaMatch->owner,betaMatch,betaMatch->counterf); 
      
      DeletePartialMatches(theEnv,betaMatch);
      
      betaMatch = tempMatch;     
     }
  }
  
/***************************************************************/
/* NegEntryRetractAlpha:           */
/***************************************************************/
static void NegEntryRetractAlpha(
  void *theEnv,
  struct partialMatch *alphaMatch,
  void *duringRetract)
  {
   struct partialMatch *betaMatch;
   struct joinNode *joinPtr;
   
   betaMatch = alphaMatch->children;
   while (betaMatch != NULL)
     {
      joinPtr = (struct joinNode *) betaMatch->owner;
      
      if (! joinPtr->patternIsNegated)
        {               
         SystemError(theEnv,"RETRACT",117);
         betaMatch = betaMatch->nextInRight;
         continue;
        }

      NegEntryRetractBeta(theEnv,joinPtr,alphaMatch,betaMatch,duringRetract);
      betaMatch = alphaMatch->children;
     }
  }

/***************************************************************/
/* NegEntryRetractBeta:           */
/***************************************************************/
static void NegEntryRetractBeta(
  void *theEnv,
  struct joinNode *joinPtr,
  struct partialMatch *alphaMatch,
  struct partialMatch *betaMatch,
  void *duringRetract)
  {
   struct joinNode *listOfJoins;
   struct rdriveinfo *tempDR;

   /*======================================================*/
   /* Try to find another RHS partial match which prevents */
   /* the LHS partial match from being satisifed.          */
   /*======================================================*/

   betaMatch->binds[betaMatch->bcount - 1].gm.theValue = NULL;
   UnlinkBetaPMFromRight(betaMatch);
   if (FindNextConflictingAlphaMatch(theEnv,betaMatch,alphaMatch->nextInNode,joinPtr,alphaMatch))
     { return; }
      
   /*=========================================================*/
   /* If the LHS partial match now has no RHS partial matches */
   /* that conflict with it, then it satisfies the conditions */
   /* of the RHS not CE. Create a partial match and send it   */
   /* to the joins below.                                     */
   /*=========================================================*/
      
   /*===============================*/
   /* Create the new partial match. */
   /*===============================*/

   betaMatch->counterf = FALSE;
   betaMatch->binds[betaMatch->bcount - 1].gm.theMatch = NULL; /* TBD Redundant? */
   UnlinkBetaPMFromNode(joinPtr,betaMatch,TRUE); /* Remove from notBeta. */
   LinkBetaPMToNode(joinPtr,betaMatch,FALSE);    /* Add to beta. */
   
   /*==============================================*/
   /* If partial matches from this join correspond */
   /* to a rule activation, then add an activation */
   /* to the agenda.                               */
   /*==============================================*/

   if (joinPtr->ruleToActivate != NULL)
     { AddActivation(theEnv,joinPtr->ruleToActivate,betaMatch); }

   /*=======================================================*/
   /* Send the partial match to the list of joins following */
   /* this join. If we're in the middle of a retract, add   */
   /* the partial match to the list of join activities that */
   /* need to be processed later. If we're doing an assert, */
   /* then the join activity can be processed immediately.  */
   /*=======================================================*/

   listOfJoins = joinPtr->nextLevel;
   if (listOfJoins == NULL) return;
      
   if (((struct joinNode *) (listOfJoins->rightSideEntryStructure)) == joinPtr)
     { NetworkAssertRightJFTR(theEnv,betaMatch,listOfJoins); }
   else
     {
      if (duringRetract != NULL)
        {
         if (FindEntityInPartialMatch((struct patternEntity *) duringRetract,betaMatch) == FALSE)
           {
            tempDR = get_struct(theEnv,rdriveinfo);
            tempDR->link = betaMatch;
            tempDR->jlist = joinPtr->nextLevel;
            tempDR->next = EngineData(theEnv)->DriveRetractionList;
            EngineData(theEnv)->DriveRetractionList = tempDR;
           }
        }
      else while (listOfJoins != NULL)
        {
         NetworkAssertLeft(theEnv,betaMatch,listOfJoins);
         listOfJoins = listOfJoins->rightDriveNode;
        }
     }
  }

/***************************************************************/
/* PosEntryRetractBeta:           */
/***************************************************************/
void PosEntryRetractBeta(
  void *theEnv,
  struct partialMatch *parentMatch,
  struct partialMatch *betaMatch,
  void *duringRetract)
  {
   struct partialMatch *tempMatch;
   struct joinNode *joinPtr;
   struct rdriveinfo *parentList, *tempDR;
   int popParent = FALSE, followSiblings;
 
   if (betaMatch == NULL) return; /* TBD Try removing later... */
   
   parentList = get_struct(theEnv,rdriveinfo);
   parentList->jlist = (struct joinNode *) parentMatch->owner;
   parentList->link = parentMatch;
   parentList->next = NULL;
   
   while (betaMatch != NULL)
     {
      /*================================================*/
      /* Visit the children before visiting the parent. */
      /*================================================*/
      
      if (betaMatch->children != NULL)
        {
         tempDR  = get_struct(theEnv,rdriveinfo);
         tempDR->jlist = (struct joinNode *) betaMatch->owner;
         tempDR->next = parentList;
         tempDR->link = betaMatch;
         parentList = tempDR;
         
         betaMatch = betaMatch->children;
         continue;
        }

      joinPtr = (struct joinNode *) betaMatch->owner;

      if ((joinPtr->joinFromTheRight) &&
          (parentList->jlist == (struct joinNode *) joinPtr->rightSideEntryStructure))
        { followSiblings = FALSE; }
      else
        { followSiblings = TRUE; }
        
      /*===============================*/
      /* Determine what to visit next. */
      /*===============================*/

      if ((betaMatch->nextInBeta != NULL) && followSiblings)
        { tempMatch = betaMatch->nextInBeta; }
      else
        {
         popParent = TRUE; 
         tempMatch = parentList->link;
        }

      if ((joinPtr->joinFromTheRight) &&
          (parentList->jlist == (struct joinNode *) joinPtr->rightSideEntryStructure))
        {
         NegEntryRetractBeta(theEnv,joinPtr,parentList->link,betaMatch,duringRetract);
        }
      else
        {
         if ((betaMatch->activationf) ?
             (betaMatch->binds[betaMatch->bcount].gm.theValue != NULL) : FALSE)
           { RemoveActivation(theEnv,(struct activation *) betaMatch->binds[betaMatch->bcount].gm.theValue,TRUE,TRUE); }
      
         UnlinkBetaPMFromNodeAndLineage(betaMatch->owner,betaMatch,betaMatch->counterf); 
      
         DeletePartialMatches(theEnv,betaMatch);
        }
      
      /*============================*/
      /* Move on to the next match. */
      /*============================*/

      if (popParent)
        {
         tempDR = parentList;
         parentList = parentList->next;
         rtn_struct(theEnv,rdriveinfo,tempDR);
         popParent = FALSE;
        }
  
      if (tempMatch == parentMatch) return;
      betaMatch = tempMatch;      
     }
  }

/**************************************************************/
/* FindNextConflictingAlphaMatch: Finds the next conflicting  */
/*   partial match in the alpha memory of a join (or the beta */
/*   memory of a join from the right) that prevents a partial */
/*   match in the beta memory of the join from being          */
/*   satisfied.                                               */
/**************************************************************/
static intBool FindNextConflictingAlphaMatch(
  void *theEnv,
  struct partialMatch *theBind,
  struct partialMatch *possibleConflicts,
  struct joinNode *theJoin,
  struct partialMatch *skipMatch)
  {
   int i, result;

   /*=====================================================*/
   /* If we're dealing with a join from the right, then   */
   /* we need to check the entire beta memory of the join */
   /* from the right (a join doesn't have an end of queue */
   /* pointer like a pattern data structure has).         */
   /*=====================================================*/

   if (theJoin->joinFromTheRight)
     { possibleConflicts = ((struct joinNode *) theJoin->rightSideEntryStructure)->beta; }

   /*====================================*/
   /* Check each of the possible partial */
   /* matches which could conflict.      */
   /*====================================*/

#if DEVELOPER
   if (possibleConflicts != NULL)
     { EngineData(theEnv)->leftToRightLoops++; }
#endif     

   for (;
        possibleConflicts != NULL;
        possibleConflicts = possibleConflicts->nextInNode)
     {
      /*=====================================*/
      /* Initially indicate that the partial */
      /* match doesn't conflict.             */
      /*=====================================*/

      result = FALSE;

      if (skipMatch == possibleConflicts)
        { /* Do Nothing */ }
        
       /*======================================================*/
       /* 6.05 Bug Fix. It is possible that a pattern entity   */
       /* (e.g., instance) in a partial match is 'out of date' */
       /* with respect to the lazy evaluation scheme use by    */
       /* negated patterns. In other words, the object may     */
       /* have changed since it was last pushed through the    */
       /* network, and thus the partial match may be invalid.  */
       /* If so, the partial match must be ignored here.       */
       /*======================================================*/

      else if (PartialMatchDefunct(theEnv,possibleConflicts))
        { /* Do Nothing */ }

      /*==================================================*/
      /* If the join doesn't have a network expression to */
      /* be evaluated, then partial match conflicts. If   */
      /* the partial match is retrieved from a join from  */
      /* the right, the RHS partial match must correspond */
      /* to the partial match in the beta memory of the   */
      /* join being examined (in a join associated with a */
      /* not CE, each partial match in the beta memory of */
      /* the join corresponds uniquely to a partial match */
      /* in either the alpha memory from the RHS or in    */
      /* the beta memory of a join from the right).       */
      /*==================================================*/

      else if (theJoin->networkTest == NULL)
        {
         result = TRUE;
         if (theJoin->joinFromTheRight)
           {
            for (i = 0; i < (int) (theBind->bcount - 1); i++)
              {
               if (possibleConflicts->binds[i].gm.theMatch != theBind->binds[i].gm.theMatch)
                 {
                  result = FALSE;
                  break;
                 }
              }
           }
        }

      /*=================================================*/
      /* Otherwise, if the join has a network expression */
      /* to evaluate, then evaluate it.                  */
      /*=================================================*/

      else
        {
#if DEVELOPER
         if (theJoin->networkTest)
           { 
            EngineData(theEnv)->leftToRightComparisons++; 
            EngineData(theEnv)->findNextConflictingComparisons++; 
           }
#endif
         result = EvaluateJoinExpression(theEnv,theJoin->networkTest,theBind,
                                         possibleConflicts,theJoin);
         if (EvaluationData(theEnv)->EvaluationError)
           {
            result = TRUE;
            EvaluationData(theEnv)->EvaluationError = FALSE;
           }
           
#if DEVELOPER
         if (result != FALSE)
           { EngineData(theEnv)->leftToRightSucceeds++; }
#endif
        }

      /*==============================================*/
      /* If the network expression evaluated to TRUE, */
      /* then partial match being examined conflicts. */
      /* Point the beta memory partial match to the   */
      /* conflicting partial match and return TRUE to */
      /* indicate a conflict was found.               */
      /*==============================================*/

      if (result != FALSE)
        {
         theBind->binds[theBind->bcount - 1].gm.theValue = (void *) possibleConflicts;
         UpdatePMAlphaLinks(theBind,possibleConflicts);
         return(TRUE);
        }
     }

   /*========================*/
   /* No conflict was found. */
   /*========================*/

   return(FALSE);
  }

/***********************************************************/
/* PartialMatchDefunct: Determines if any pattern entities */
/*   contained within the partial match have changed since */
/*   this partial match was generated. Assumes counterf is */
/*   FALSE.                                                */
/***********************************************************/
static intBool PartialMatchDefunct(
  void *theEnv,
  struct partialMatch *thePM)
  {
   register unsigned i;
   register struct patternEntity * thePE;

   for (i = 0 ; i < thePM->bcount ; i++)
     {
      if (thePM->binds[i].gm.theMatch == NULL) continue;
      thePE = thePM->binds[i].gm.theMatch->matchingItem;
      if (thePE && thePE->theInfo->synchronized &&
          !(*thePE->theInfo->synchronized)(theEnv,thePE))
        return(TRUE);
     }
   return(FALSE);
  }

/***************************************************/
/* DeletePartialMatches: Returns a list of partial */
/*   matches to the pool of free memory.           */
/***************************************************/
void DeletePartialMatches(
  void *theEnv,
  struct partialMatch *listOfPMs)
  {
   struct partialMatch *nextPM;

   while (listOfPMs != NULL)
     {
      /*============================================*/
      /* Remember the next partial match to delete. */
      /*============================================*/

      nextPM = listOfPMs->nextInNode;

      /*================================================*/
      /* Remove the links between the partial match and */
      /* any data entities that it is attached to as a  */
      /* result of a logical CE.                        */
      /*================================================*/

      if (listOfPMs->dependentsf) RemoveLogicalSupport(theEnv,listOfPMs);

      /*==========================================================*/
      /* If the partial match is being deleted from a beta memory */
      /* and the partial match isn't associated with a satisfied  */
      /* not CE, then it can be immediately returned to the pool  */
      /* of free memory. Otherwise, it's could be in use (either  */
      /* to retrieve variables from the LHS or by the activation  */
      /* of the rule). Since a not CE creates a "pseudo" data     */
      /* entity, the beta partial match which stores this pseudo  */
      /* data entity can not be deleted immediately (for the same */
      /* reason an alpha memory partial match can't be deleted    */
      /* immediately).                                            */
      /*==========================================================*/

      ReturnPartialMatch(theEnv,listOfPMs);

      /*====================================*/
      /* Move on to the next partial match. */
      /*====================================*/

      listOfPMs = nextPM;
     }
  }

/**************************************************************/
/* ReturnPartialMatch: Returns the data structures associated */
/*   with a partial match to the pool of free memory.         */
/**************************************************************/
globle void ReturnPartialMatch(
  void *theEnv,
  struct partialMatch *waste)
  {
   /*==============================================*/
   /* If the partial match is in use, then put it  */
   /* on a garbage list to be processed later when */
   /* the partial match is not in use.             */
   /*==============================================*/

   if (waste->busy)
     {
      waste->nextInNode = EngineData(theEnv)->GarbagePartialMatches;
      EngineData(theEnv)->GarbagePartialMatches = waste;
      return;
     }

   /*======================================================*/
   /* If we're dealing with an alpha memory partial match, */
   /* then return the multifield markers associated with   */
   /* the partial match (if any) along with the alphaMatch */
   /* data structure.                                      */
   /*======================================================*/

   if (waste->betaMemory == FALSE)
     {
      if (waste->binds[0].gm.theMatch->markers != NULL)
        { ReturnMarkers(theEnv,waste->binds[0].gm.theMatch->markers); }
      rm(theEnv,waste->binds[0].gm.theMatch,(int) sizeof(struct alphaMatch));
     }

   /*============================================*/
   /* Return the pseudo fact associated with the */
   /* not CE referenced by the partial match.    */
   /*============================================*/

   if ((waste->notOriginf) &&
       (waste->counterf == FALSE) &&
       (waste->binds[waste->bcount - 1].gm.theMatch != NULL))
      { rtn_struct(theEnv,alphaMatch,waste->binds[waste->bcount - 1].gm.theMatch); }

   /*=================================================*/
   /* Remove any links between the partial match and  */
   /* a data entity that were created with the use of */
   /* the logical CE.                                 */
   /*=================================================*/

   if (waste->dependentsf) RemovePMDependencies(theEnv,waste);

   /*======================================================*/
   /* Return the partial match to the pool of free memory. */
   /*======================================================*/

   rtn_var_struct(theEnv,partialMatch,(int) sizeof(struct genericMatch *) *
                  (waste->bcount + waste->activationf + waste->dependentsf - 1),
                  waste);
  }

/***************************************************************/
/* DestroyPartialMatch: Returns the data structures associated */
/*   with a partial match to the pool of free memory.          */
/***************************************************************/
globle void DestroyPartialMatch(
  void *theEnv,
  struct partialMatch *waste)
  {
   /*======================================================*/
   /* If we're dealing with an alpha memory partial match, */
   /* then return the multifield markers associated with   */
   /* the partial match (if any) along with the alphaMatch */
   /* data structure.                                      */
   /*======================================================*/

   if (waste->betaMemory == FALSE)
     {
      if (waste->binds[0].gm.theMatch->markers != NULL)
        { ReturnMarkers(theEnv,waste->binds[0].gm.theMatch->markers); }
      rm(theEnv,waste->binds[0].gm.theMatch,(int) sizeof(struct alphaMatch));
     }
     
   /*=================================================*/
   /* Remove any links between the partial match and  */
   /* a data entity that were created with the use of */
   /* the logical CE.                                 */
   /*=================================================*/

   if (waste->dependentsf) DestroyPMDependencies(theEnv,waste);

   /*======================================================*/
   /* Return the partial match to the pool of free memory. */
   /*======================================================*/

   rtn_var_struct(theEnv,partialMatch,(int) sizeof(struct genericMatch *) *
                  (waste->bcount + waste->activationf + waste->dependentsf - 1),
                  waste);
  }

/******************************************************/
/* ReturnMarkers: Returns a linked list of multifield */
/*   markers associated with a data entity matching a */
/*   pattern to the pool of free memory.              */
/******************************************************/
static void ReturnMarkers(
  void *theEnv,
  struct multifieldMarker *waste)
  {
   struct multifieldMarker *temp;

   while (waste != NULL)
     {
      temp = waste->next;
      rtn_struct(theEnv,multifieldMarker,waste);
      waste = temp;
     }
  }

/*************************************************/
/* DriveRetractions: Filters the list of partial */
/*   matches created as a result of removing a   */
/*   data entity through the join network.       */
/*************************************************/
static void DriveRetractions(
  void *theEnv)
  {
   struct rdriveinfo *tempDR;
   struct joinNode *joinPtr;

   while (EngineData(theEnv)->DriveRetractionList != NULL)
     {
      for (joinPtr = EngineData(theEnv)->DriveRetractionList->jlist;
           joinPtr != NULL;
           joinPtr = joinPtr->rightDriveNode)
        { NetworkAssertLeft(theEnv,EngineData(theEnv)->DriveRetractionList->link,joinPtr); }

      tempDR = EngineData(theEnv)->DriveRetractionList->next;
      rtn_struct(theEnv,rdriveinfo,EngineData(theEnv)->DriveRetractionList);
      EngineData(theEnv)->DriveRetractionList = tempDR;
     }
  }

/*************************************************/
/* RetractCheckDriveRetractions:       */
/*************************************************/
globle void RetractCheckDriveRetractions(  /* GDR 111599 #834 Begin */
  void *theEnv,
  struct alphaMatch *theAlphaNode,
  int position)
  {
   struct rdriveinfo *tempDR, *theDR, *lastDR = NULL;

   theDR = EngineData(theEnv)->DriveRetractionList;
   while (theDR != NULL)
     {
      if ((position < (int) theDR->link->bcount) &&
          (theDR->link->binds[position].gm.theMatch == theAlphaNode))
        {
         tempDR = theDR->next;
         rtn_struct(theEnv,rdriveinfo,theDR);
         if (lastDR == NULL)
           { EngineData(theEnv)->DriveRetractionList = tempDR; }
         else
           { lastDR->next = tempDR; }
         theDR = tempDR;
        }
      else
        {
         lastDR = theDR;
         theDR = theDR->next;
        }
     }
  }                                        /* GDR 111599 #834 End */
  
/*************************************************************/
/* FlushGarbagePartialMatches:  Returns partial matches and  */
/*   associated structures that were removed as part of a    */
/*   retraction. It is necessary to postpone returning these */
/*   structures to memory because RHS actions retrieve their */
/*   variable bindings directly from the fact and instance   */
/*   data structures through the alpha memory bindings.      */
/*************************************************************/
globle void FlushGarbagePartialMatches(
  void *theEnv)
  {
   struct partialMatch *pmPtr;
   struct alphaMatch *amPtr;

   /*===================================================*/
   /* Return the garbage partial matches collected from */
   /* the alpha memories of the pattern networks.       */
   /*===================================================*/

   while (EngineData(theEnv)->GarbageAlphaMatches != NULL)
     {
      amPtr = EngineData(theEnv)->GarbageAlphaMatches->next;
      rtn_struct(theEnv,alphaMatch,EngineData(theEnv)->GarbageAlphaMatches);
      EngineData(theEnv)->GarbageAlphaMatches = amPtr;
     }

   /*==============================================*/
   /* Return the garbage partial matches collected */
   /* from the beta memories of the join networks. */
   /*==============================================*/

   while (EngineData(theEnv)->GarbagePartialMatches != NULL)
     {
      /*=====================================================*/
      /* Remember the next garbage partial match to process. */
      /*=====================================================*/

      pmPtr = EngineData(theEnv)->GarbagePartialMatches->nextInNode;

      /*============================================*/
      /* Dispose of the garbage partial match being */
      /* examined and move on to the next one.      */
      /*============================================*/

      EngineData(theEnv)->GarbagePartialMatches->busy = FALSE;
      ReturnPartialMatch(theEnv,EngineData(theEnv)->GarbagePartialMatches);
      EngineData(theEnv)->GarbagePartialMatches = pmPtr;
     }
  }

#endif /* DEFRULE_CONSTRUCT */

